///////////////////////////////////////////////////////////////////////////////
// StlModel.cpp
// ============
// STL (STereo Lithography) 3D model loader
// STL can be defined either ASCII or Binary format.
// 1) ASCII Format
//  - ASCII file must begin with "solid"
//  - Each facet begins with "facet" and follows normal
//      facet normal ...
//          outer loop
//              vertex ...
//              vertex ...
//              vertex ...
//          endloop
//      endfacet
//
// 2) Binary Format
//  - Binary file begins with 80-byte header
//  - 4-byte (int) for the # of facets
//  - 50-byte for each facet = 12-byte normal + 36-byte vertices + 2-byte padding
//
//  AUTHOR: Song Ho Ahn (song.ahn@gmail.com)
// CREATED: 2025-09-19
// UPDATED: 2025-12-06
///////////////////////////////////////////////////////////////////////////////

#include "StlModel.h"
#include "Tokenizer.h"
#include <fstream>
#include <iostream>
#include <sstream>
#include <vector>
#include <map>
#include <algorithm>
#include <limits>
#include <ctime>


// constants
const float EPSILON = 0.00001f;



///////////////////////////////////////////////////////////////////////////////
// ctor
///////////////////////////////////////////////////////////////////////////////
StlModel::StlModel() : errorMessage("No Error.")
{
}



///////////////////////////////////////////////////////////////////////////////
// dtor
///////////////////////////////////////////////////////////////////////////////
StlModel::~StlModel()
{
}



///////////////////////////////////////////////////////////////////////////////
// print itself
///////////////////////////////////////////////////////////////////////////////
void StlModel::printSelf() const
{
    std::cout << "===== StlModel =====\n"
              << "Triangle Count: " << getTriangleCount() << "\n"
              << "   Index Count: " << getIndexCount() << "\n"
              << "  Bounding Box: " << bound.toString() << "\n"
              << "  Vertex Count: " << getVertexCount() << "\n"
              << "  Normal Count: " << getNormalCount() << std::endl;

    std::cout << std::endl;
}



///////////////////////////////////////////////////////////////////////////////
// init containers and vars
///////////////////////////////////////////////////////////////////////////////
void StlModel::init()
{
    stride = 0;

    // reset bounding box
    bound.set(0,0, 0,0, 0,0);

    // flush(deallocate) the previous memory
    std::vector<float>().swap(vertices);
    std::vector<float>().swap(normals);

    std::vector<unsigned int>().swap(indices);
    std::vector<Vector3>().swap(faceNormals);
    std::vector<float>().swap(interleavedVertices);
}



///////////////////////////////////////////////////////////////////////////////
// load STL file from file
///////////////////////////////////////////////////////////////////////////////
bool StlModel::read(const char* fileName)
{
    // validate file name
    if(!fileName)
    {
        errorMessage = "File name is not defined.";
        return false;
    }

    // remember path and file name (assume fileName has absolute path)
    std::string path = fileName;
    std::size_t index = path.find_last_of("/\\");
    if(index != std::string::npos)
    {
        stlDirectory = path.substr(0, index+1);
        stlFileName = path.substr(index+1);
    }
    else
    {
        stlDirectory = "";
        stlFileName = fileName;
    }
    path = stlDirectory + stlFileName;  // full path (dir + file)

    // open an STL file
    std::ifstream inFile;
    inFile.open(path.c_str(), std::ios::in);
    if(!inFile.good())
    {
        errorMessage = "Failed to open a STL file to read: ";
        errorMessage += path;
        inFile.close();
        return false;
    }

    // check ascii or binary format
    // NOTE: ASCII format must begin with "solid"
    const int BUFFER_SIZE = 5;
    char* b = new char[BUFFER_SIZE];
    inFile.read(b, BUFFER_SIZE);
    inFile.close();

    if(b[0] == 's' && b[1] == 'o' && b[2] == 'l' && b[3] == 'i' && b[4] == 'd')
        return readAscii(fileName);
    else
        return readBinary(fileName);
}



///////////////////////////////////////////////////////////////////////////////
// ASCII format structure is
// solid
// facet normal
//   outer loop
//     vertex
//     vertex
//     vertex
//   endloop
// endfacet
// endsolid
///////////////////////////////////////////////////////////////////////////////
bool StlModel::readAscii(const char* fileName)
{
    std::ifstream inFile;
    inFile.open(fileName, std::ios::in);
    if(!inFile.good())
    {
        errorMessage = "Failed to open a STL ASCII file to read.";
        return false;            // exit if failed
    }

    // init arrays for opengl drawing
    std::vector<unsigned int>().swap(indices);
    std::vector<Vector3>().swap(faceNormals);
    std::vector<float>().swap(interleavedVertices);
    std::vector<float>().swap(vertices);
    std::vector<float>().swap(normals);

    // get lines of STL file
    float nx, ny, nz;
    unsigned int faceIndex = 0;
    Tokenizer tokenizer;
    std::vector<std::string> tokens;
    std::string line;
    while(std::getline(inFile, line))
    {
        // parse a face
        if(line.find("facet") != std::string::npos)
        {
            tokenizer.set(line, " ");
            tokens = tokenizer.split();
            if(tokens.size() > 4)
            {
                nx = s2f(tokens[2]);
                ny = s2f(tokens[3]);
                nz = s2f(tokens[4]);
            }
            else
            {
                nx = 0;
                ny = 0;
                nz = 1;
            }
            faceNormals.push_back(Vector3(nx, ny, nz));

            // add vertices
            while(std::getline(inFile, line))
            {
                if(line.find("vertex") != std::string::npos)
                {
                    tokenizer.set(line, " ");
                    tokens = tokenizer.split();
                    if(tokens.size() > 3)
                    {
                        vertices.push_back(s2f(tokens[1]));
                        vertices.push_back(s2f(tokens[2]));
                        vertices.push_back(s2f(tokens[3]));
                        normals.push_back(nx);
                        normals.push_back(ny);
                        normals.push_back(nz);
                        indices.push_back(faceIndex++);
                    }
                }
                else if(line.find("endfacet") != std::string::npos)
                {
                    break; // exit facet loop
                }
            }
        }
    }

    // close opened file
    inFile.close();

    // remove duplicate vertices
    removeDuplicates();

    // construct interleaved array
    buildInterleavedVertices();

    // compute bounding box
    computeBoundingBox();

    return true;
}



///////////////////////////////////////////////////////////////////////////////
// BINARY format structure is
// 80 bytes: file header
//  4 bytes: # of facets (int)
// 50 bytes: each facet = 12 bytes normal + 36 bytes vertex + 2 bytes padding
///////////////////////////////////////////////////////////////////////////////
bool StlModel::readBinary(const char* fileName)
{
    std::ifstream inFile;
    inFile.open(fileName, std::ios::binary);
    if(!inFile.good())
    {
        errorMessage = "Failed to open a STL binary file to read.";
        return false;            // exit if failed
    }

    // init arrays for opengl drawing
    std::vector<unsigned int>().swap(indices);
    std::vector<Vector3>().swap(faceNormals);
    std::vector<float>().swap(interleavedVertices);
    std::vector<float>().swap(vertices);
    std::vector<float>().swap(normals);
    unsigned int faceIndex = 0;

    char header[80];
    int faceCount;      // 4-byte
    float f[12];        // n + v1 + v2 + v3
    char space[2];      // padding
    inFile.read(header, 80);
    inFile.read((char*)&faceCount, 4);

    for(int i = 0; i < faceCount; ++i)
    {
        inFile.read((char*)&f, 12*4);

        faceNormals.push_back(Vector3(f[0], f[1], f[2]));

        vertices.push_back(f[3]);   // v1
        vertices.push_back(f[4]);
        vertices.push_back(f[5]);
        vertices.push_back(f[6]);   // v2
        vertices.push_back(f[7]);
        vertices.push_back(f[8]);
        vertices.push_back(f[9]);   // v3
        vertices.push_back(f[10]);
        vertices.push_back(f[11]);

        for(int j = 0; j < 3; ++j)
        {
            normals.push_back(f[0]);
            normals.push_back(f[1]);
            normals.push_back(f[2]);
        }

        indices.push_back(faceIndex++);
        indices.push_back(faceIndex++);
        indices.push_back(faceIndex++);

        inFile.read((char*)&space, 2);  // consume space
    }

    // close opened file
    inFile.close();

    // remove duplicated vertices and rebuild interleaved array
    removeDuplicates();

    // construct interleaved array
    buildInterleavedVertices();

    // compute bounding box
    computeBoundingBox();

    return true;
}



///////////////////////////////////////////////////////////////////////////////
// compute bounding box of the object
///////////////////////////////////////////////////////////////////////////////
void StlModel::computeBoundingBox()
{
    // prepare default bound with opposite values
    bound.minX = bound.minY = bound.minZ = std::numeric_limits<float>::max();
    bound.maxX = bound.maxY = bound.maxZ = -std::numeric_limits<float>::max();

    float x, y, z;
    unsigned int count = (unsigned int)vertices.size();
    for(unsigned int i = 0; i < count; i += 3)
    {
        x = vertices[i];
        y = vertices[i+1];
        z = vertices[i+2];

        if(x < bound.minX) bound.minX = x;
        if(x > bound.maxX) bound.maxX = x;

        if(y < bound.minY) bound.minY = y;
        if(y > bound.maxY) bound.maxY = y;

        if(z < bound.minZ) bound.minZ = z;
        if(z > bound.maxZ) bound.maxZ = z;
    }
}



///////////////////////////////////////////////////////////////////////////////
// First, make all faces are split (not shared) and assign face normal to
// vertex. Next find shared faces and average the face normals to soften/harden.
// Finally, join (weld) shared vertices into one.
// Note that this process increase the number of vertices because it
// will convert shared vertices to independent.
///////////////////////////////////////////////////////////////////////////////
void StlModel::smoothNormals(float angle)
{
    // clean up the previous
    std::vector<Vector3>().swap(splitVertices);
    std::vector<Vector3>().swap(splitNormals);
    splitVertexMap.clear();
    sharedVertexLookup.clear();

    // a vertex on the hard edge should have separate vertex attributes at the
    // shared vertex in OpenGL because position and normal must be paired in
    // order to be drawn, so, we force to split faces for the hard edge faces.
    splitFaces();

    // copy face normal to the vertex normals, flatten(harden) the surface normals
    std::size_t indexCount = indices.size();
    for(std::size_t i = 0; i < indexCount; i += 3)
    {
        Vector3 faceNormal;
        faceNormal = faceNormals[i / 3];
        for(int j = 0; j < 3; ++j)
            splitNormals[i+j] = faceNormal;
    }

    // soften vertex normals at shared vertices
    averageNormals(angle);

    // join shared vertices only if all vertex attributes are same far each
    // vertex. It will reduce the number of vertex attributes.
    joinFaces();

    buildInterleavedVertices();

    // return memory to system after smoothing
    std::vector<Vector3>().swap(splitVertices);
    std::vector<Vector3>().swap(splitNormals);
    splitVertexMap.clear();
    sharedVertexLookup.clear();
}



///////////////////////////////////////////////////////////////////////////////
// remove the duplicated vertices
///////////////////////////////////////////////////////////////////////////////
void StlModel::removeDuplicates()
{
    // clean up the previous
    std::vector<Vector3>().swap(splitVertices);
    std::vector<Vector3>().swap(splitNormals);
    splitVertexMap.clear();
    sharedVertexLookup.clear();

    // make each face independent (not shared)
    splitFaces();

    // find same vertex data then store the shared vertices to sharedVertexLookup
    findDuplicates();

    // weld same vertex together
    joinFaces();

    buildInterleavedVertices();

    // return memory to system after merging
    std::vector<Vector3>().swap(splitVertices);
    std::vector<Vector3>().swap(splitNormals);
    splitVertexMap.clear();
    sharedVertexLookup.clear();
}



///////////////////////////////////////////////////////////////////////////////
// split faces before regenerating normals
///////////////////////////////////////////////////////////////////////////////
void StlModel::splitFaces()
{
    Vector3 vertex;
    Vector3 normal;
    unsigned int vertexIndex;

    // reserve container size
    unsigned int indexCount = (unsigned int)indices.size();
    splitVertices.reserve(indexCount);
    splitNormals.reserve(indexCount);

    // copy vertex attributes, but split them as not shared
    for(unsigned int i = 0; i < indexCount; i += 3)
    {
        // get 3 vertices and normals of a face
        vertexIndex = indices[i] * 3;
        vertex.set(vertices[vertexIndex], vertices[vertexIndex+1], vertices[vertexIndex+2]);
        splitVertices.push_back(vertex);
        splitVertexMap.insert(std::pair<Vector3, unsigned int>(vertex, i));

        normal.set(normals[vertexIndex], normals[vertexIndex+1], normals[vertexIndex+2]);
        splitNormals.push_back(normal);

        vertexIndex = indices[i+1] * 3;
        vertex.set(vertices[vertexIndex], vertices[vertexIndex+1], vertices[vertexIndex+2]);
        splitVertices.push_back(vertex);
        splitVertexMap.insert(std::pair<Vector3, unsigned int>(vertex, i+1));

        normal.set(normals[vertexIndex], normals[vertexIndex+1], normals[vertexIndex+2]);
        splitNormals.push_back(normal);

        vertexIndex = indices[i+2] * 3;
        vertex.set(vertices[vertexIndex], vertices[vertexIndex+1], vertices[vertexIndex+2]);
        splitVertices.push_back(vertex);
        splitVertexMap.insert(std::pair<Vector3, unsigned int>(vertex, i+2));

        normal.set(normals[vertexIndex], normals[vertexIndex+1], normals[vertexIndex+2]);
        splitNormals.push_back(normal);
    }
}



///////////////////////////////////////////////////////////////////////////////
// find shared vertices that their normals are less than smooth angle, then
// soften the vertex normals by averaging their face normals
// This routine uses the multimap prepared in splitFaces() in order to speed up
// the averaging algorithm.
///////////////////////////////////////////////////////////////////////////////
void StlModel::averageNormals(float angle)
{
    typedef std::multimap<Vector3, unsigned int>::iterator MapIter;

    const float DEG2RAD = acos(-1.0f) / 180.0f;
    float cosAngle = cosf(DEG2RAD * angle);

    // first, find unique vertex keys
    std::vector<Vector3> vertexKeys;
    MapIter mapIter = splitVertexMap.begin();

    // add the first one into vertex key list
    Vector3 vertexKey = mapIter->first;
    vertexKeys.push_back(vertexKey);
    ++mapIter;

    // find different keys from the map
    while(mapIter != splitVertexMap.end())
    {
        // add new key to the list if found different key
        if(vertexKey != mapIter->first)
        {
            vertexKey = mapIter->first;     // remember new key for next search
            vertexKeys.push_back(vertexKey);
        }
        ++mapIter;
    }

    // init shared vertex index map with itself
    unsigned int vertexCount = (unsigned int)splitVertices.size();
    sharedVertexLookup.clear();
    for(unsigned int i = 0; i < vertexCount; ++i)
        sharedVertexLookup[i] = i;

    // do average
    Vector3 sharedNormal, normal1, normal2;
    unsigned int vertexIndex1, vertexIndex2;

    // loop through each vertex
    std::vector<Vector3>::iterator iter;
    for(iter = vertexKeys.begin(); iter != vertexKeys.end(); ++iter)
    {
        // get all indices with same vertex position
        std::pair<MapIter, MapIter> iterRange = splitVertexMap.equal_range(*iter);

        // get normal of the first element
        mapIter = iterRange.first;
        vertexIndex1 = mapIter->second;
        sharedNormal = normal1 = splitNormals[vertexIndex1];

        // compare with other vertex normals
        ++mapIter;
        while(mapIter != iterRange.second)
        {
            vertexIndex2 = mapIter->second;
            //faceIndex2 = vertexIndex2 / 3;
            normal2 = splitNormals[vertexIndex2];

            // check if 2 normals are less than smooth angle
            if(normal1.dot(normal2) > cosAngle)
            {
                // remember the shared vertex index, so we can join the vertex together
                sharedVertexLookup[vertexIndex2] = vertexIndex1;

                sharedNormal += normal2;    // sum normal
            }
            ++mapIter;
        }
        sharedNormal.normalize();   // make it unit length
        splitNormals[vertexIndex1] = sharedNormal;
    }
}



///////////////////////////////////////////////////////////////////////////////
// find same vertices where position, normal and texCoord are same
///////////////////////////////////////////////////////////////////////////////
void StlModel::findDuplicates()
{
    typedef std::multimap<Vector3, unsigned int>::iterator MapIter;

    // first, find unique vertex keys
    MapIter mapIter = splitVertexMap.begin();

    // add the first one into vertex key list
    std::vector<Vector3> vertexKeys;
    Vector3 vertexKey = mapIter->first;
    vertexKeys.push_back(vertexKey);
    ++mapIter;

    // find all different keys from the map
    while(mapIter != splitVertexMap.end())
    {
        // add new key to the list if the key is different
        if(vertexKey != mapIter->first)
        {
            vertexKey = mapIter->first;     // remember new key for next search
            vertexKeys.push_back(vertexKey);
        }
        ++mapIter;
    }

    // init shared vertex index map with itself
    sharedVertexLookup.clear();
    unsigned int vertexCount = (unsigned int)splitVertices.size();
    for(unsigned int i = 0; i < vertexCount; ++i)
        sharedVertexLookup[i] = i;

    Vector3 normal1, normal2;
    unsigned int index1, index2;

    // loop through each vertex to compare other vertices
    std::vector<Vector3>::iterator iter;
    for(iter = vertexKeys.begin(); iter != vertexKeys.end(); ++iter)
    {
        // get all vertex indices with same vertex position
        std::pair<MapIter, MapIter> iterRange = splitVertexMap.equal_range(*iter);

        // remember index range that has same vertex position
        std::vector<unsigned int> indexRange;
        for(mapIter = iterRange.first; mapIter != iterRange.second; ++mapIter)
            indexRange.push_back(mapIter->second);

        unsigned int rangeCount = (unsigned int)indexRange.size();
        for(unsigned int i = 0; i < rangeCount; ++i)
        {
            // get the normal of the first element
            index1 = indexRange[i];
            normal1 = splitNormals[index1];

            // compare with other vertex normals
            for(unsigned int j = i+1; j < rangeCount; ++j)
            {
                index2 = indexRange[j];
                normal2 = splitNormals[index2];

                // check if 2 normals are same
                if(normal1 == normal2)
                {
                    // remember this shared vertex index, so we can join the vertex together
                    // If the index and value of lookup are different, then it is
                    // already set as duplicate. So, we update only where index and
                    // value are same.
                    if(sharedVertexLookup[index2] == index2)
                        sharedVertexLookup[index2] = index1;
                }
            }
        }
    }
}



///////////////////////////////////////////////////////////////////////////////
// weld shared vertices together
///////////////////////////////////////////////////////////////////////////////
void StlModel::joinFaces()
{
    Vector3 vec3;
    Vector2 vec2;
    unsigned int index;
    unsigned int vertexCount = (unsigned int)splitVertices.size();

    std::map<unsigned int, unsigned int> newIndexLookup;

    // clear previous lists
    vertices.clear();
    normals.clear();
    indices.clear();

    // loop through all vertices
    for(unsigned int i = 0; i < vertexCount; ++i)
    {
        // if index is same, then not shared, add as new vertex
        if(sharedVertexLookup[i] == i)
        {
            vec3 = splitVertices[i];            // add position
            vertices.push_back(vec3.x);
            vertices.push_back(vec3.y);
            vertices.push_back(vec3.z);

            vec3 = splitNormals[i];             // add normal
            normals.push_back(vec3.x);
            normals.push_back(vec3.y);
            normals.push_back(vec3.z);

            index = (unsigned int)vertices.size() / 3 - 1; // add index
            indices.push_back(index);

            newIndexLookup[i] = index;          // remember new index for other shared vertex
        }
        // shared vertex, add only index to the index list
        else
        {
            indices.push_back(newIndexLookup[sharedVertexLookup[i]]);
        }
    }
}



///////////////////////////////////////////////////////////////////////////////
// compute normal with 3 face vertices
///////////////////////////////////////////////////////////////////////////////
Vector3 StlModel::computeFaceNormal(const Vector3& v1,
                                    const Vector3& v2,
                                    const Vector3& v3)
{
    Vector3 v12 = v2 - v1;
    Vector3 v13 = v3 - v1;
    Vector3 normal = v12.cross(v13);
    normal.normalize();

    return normal;
}



///////////////////////////////////////////////////////////////////////////////
// return interleaved data
///////////////////////////////////////////////////////////////////////////////
const float* StlModel::getInterleavedVertices()
{
    // create one if not built yet
    if(interleavedVertices.size() <= 0)
        buildInterleavedVertices();

    return &interleavedVertices[0];
}



///////////////////////////////////////////////////////////////////////////////
// create interleaved vertices: v-n
// It assume the number of vertices is same as normals.
///////////////////////////////////////////////////////////////////////////////
void StlModel::buildInterleavedVertices()
{
    // compute stride
    stride = 24;    // = 4 * 6
    interleavedVertices.clear();

    unsigned int count = (unsigned int)vertices.size();
    for(unsigned int i = 0; i < count; i += 3)
    {
        interleavedVertices.push_back(vertices[i]);
        interleavedVertices.push_back(vertices[i+1]);
        interleavedVertices.push_back(vertices[i+2]);

        interleavedVertices.push_back(normals[i]);
        interleavedVertices.push_back(normals[i+1]);
        interleavedVertices.push_back(normals[i+2]);
    }
}



///////////////////////////////////////////////////////////////////////////////
// save to STL file
// If the 4x4 transform matrix is provided, thr vertex data will be transform
// before saving. But, the original vertex data are not changed.
///////////////////////////////////////////////////////////////////////////////
bool StlModel::save(const char* fileName, bool binary, const float* matrix)
{
    // validate file name
    if(!fileName)
    {
        errorMessage = "File name is not defined.";
        return false;
    }

    if(binary)
        return saveBinary(fileName, matrix);
    else
        return saveAscii(fileName, matrix);
}



///////////////////////////////////////////////////////////////////////////////
// save STL as ASCII format
///////////////////////////////////////////////////////////////////////////////
bool StlModel::saveAscii(const char* fileName, const float* matrix)
{
    float rotMat[16] = {1,0,0,0, 0,1,0,0, 0,0,1,0, 0,0,0,1};
    if(matrix)
    {
        // use only rotation part for normal transform
        rotMat[0] = matrix[0]; rotMat[1] = matrix[1]; rotMat[2] = matrix[2];
        rotMat[4] = matrix[4]; rotMat[5] = matrix[5]; rotMat[6] = matrix[6];
        rotMat[8] = matrix[8]; rotMat[9] = matrix[9]; rotMat[10]= matrix[10];
    }

    // open a file
    std::ofstream outFile;
    outFile.open(fileName);
    if(!outFile.good())
    {
        errorMessage = "Failed to open a ASCII file to save: ";
        errorMessage += fileName;
        outFile.close();
        return false;
    }

    std::stringstream ss;
    ss << "solid stl_model\n";

    std::size_t i, j, vIndex;
    Vector3 v;
    std::size_t count = indices.size();
    for(i = 0; i < count; i += 3)
    {
        vIndex = indices[i] * 3;
        v.set(normals[vIndex], normals[vIndex+1], normals[vIndex+2]);
        if(matrix)
            v = transform(rotMat, v);

        ss << " facet normal " << v.x << " " << v.y << " " << v.z << "\n";
        ss << "  outer loop\n";

        for(j = i; j < i+3; ++j)
        {
            vIndex = indices[j] * 3;
            v.set(vertices[vIndex], vertices[vIndex+1], vertices[vIndex+2]);
            if(matrix)
                v = transform(matrix, v);

            ss << "   vertex " << v.x << " " << v.y << " " << v.z << "\n";
        }

        ss << "  endloop\n";
        ss << " endfacet\n";
    }
    ss << "endsolid\n";

    outFile << ss.str();

    // close opened file
    outFile.close();
    return true;
}



///////////////////////////////////////////////////////////////////////////////
// save STL as binary format
///////////////////////////////////////////////////////////////////////////////
bool StlModel::saveBinary(const char* fileName, const float* matrix)
{
    float rotMat[16] = {1,0,0,0, 0,1,0,0, 0,0,1,0, 0,0,0,1};
    if(matrix)
    {
        // use only rotation part for normal transform
        rotMat[0] = matrix[0]; rotMat[1] = matrix[1]; rotMat[2] = matrix[2];
        rotMat[4] = matrix[4]; rotMat[5] = matrix[5]; rotMat[6] = matrix[6];
        rotMat[8] = matrix[8]; rotMat[9] = matrix[9]; rotMat[10]= matrix[10];
    }

    // open a file
    std::ofstream outFile;
    outFile.open(fileName, std::ios::binary);
    if(!outFile.good())
    {
        errorMessage = "Failed to open a binary file to save: ";
        errorMessage += fileName;
        outFile.close();
        return false;
    }

    char header[80] = "Generated by StlModel (song.ahn@gmail.com)";
    unsigned int faceCount = getTriangleCount();
    char space[2] = {'\0', '\0'};

    // write header & facet count
    outFile.write(header, 80);
    outFile.write((char*)&faceCount, 4);

    Vector3 v;
    std::size_t i, j, vIndex;
    std::size_t count = indices.size();
    for(i = 0; i < count; i += 3)
    {
        vIndex = indices[i] * 3;
        v.set(normals[vIndex], normals[vIndex+1], normals[vIndex+2]);
        if(matrix)
            v = transform(rotMat, v);

        outFile.write((char*)&v.x, 12);

        for(j = i; j < i+3; ++j)
        {
            vIndex = indices[j] * 3;
            v.set(vertices[vIndex], vertices[vIndex+1], vertices[vIndex+2]);
            if(matrix)
                v = transform(matrix, v);

            outFile.write((char*)&v.x, 12);
        }

        outFile.write(space, 2);
    }

    // close opened file
    outFile.close();
    return true;
}



///////////////////////////////////////////////////////////////////////////////
// transform a vertex data by multiplying by a 4x4 matrix
///////////////////////////////////////////////////////////////////////////////
Vector3 StlModel::transform(const float* mat, const Vector3& vec)
{
    Vector3 result;
    result.x = mat[0] * vec.x + mat[4] * vec.y + mat[8] * vec.z + mat[12];
    result.y = mat[1] * vec.x + mat[5] * vec.y + mat[9] * vec.z + mat[13];
    result.z = mat[2] * vec.x + mat[6] * vec.y + mat[10]* vec.z + mat[14];
    return result;
}



///////////////////////////////////////////////////////////////////////////////
// copy vertex data to this
///////////////////////////////////////////////////////////////////////////////
void StlModel::copy(const float* vertices, unsigned int vertexCount,
                    const float* normals, unsigned int normalCount,
                    const unsigned int* indices, unsigned int indexCount)
{
    this->vertices.resize(vertexCount * 3);
    std::copy(vertices, vertices+(vertexCount*3), this->vertices.begin());
    this->normals.resize(normalCount * 3);
    std::copy(normals, normals+(normalCount*3), this->normals.begin());
    this->indices.resize(indexCount);
    std::copy(indices, indices+indexCount, this->indices.begin());

    buildInterleavedVertices();
    computeBoundingBox();
}



///////////////////////////////////////////////////////////////////////////////
// convert string to float
///////////////////////////////////////////////////////////////////////////////
float StlModel::s2f(const std::string& s)
{
    if(s == "0" || s == "-0")
        return (float)0;
    else
        return (float)atof(s.c_str());
}
