///////////////////////////////////////////////////////////////////////////////
// StlModel.h
// ==========
// STL (STereo Lithography) 3D model loader
// STL can be defined either ASCII or Binary format.
// 1) ASCII Format
//  - ASCII file must begin with "solid"
//  - Each facet begins with "facet" and follows normal
//      facet normal ...
//          outer loop
//              vertex ...
//              vertex ...
//              vertex ...
//          endloop
//      endfacet
//
// 2) Binary Format
//  - Binary file begins with 80-byte header
//  - 4-byte (int) for the # of facets
//  - 50-byte for each facet = 12-byte normal + 36-byte vertices + 2-byte padding
//
//  AUTHOR: Song Ho Ahn (song.ahn@gmail.com)
// CREATED: 2025-09-19
// UPDATED: 2025-12-06
///////////////////////////////////////////////////////////////////////////////

#ifndef STL_MODEL_H
#define STL_MODEL_H

#include <vector>
#include <string>
#include <map>
#include <sstream>
#include "BoundingBox.h"
#include "Vectors.h"


// constants //////////////////////////////////////////////////////////////////
const float STL_SMOOTH_ANGLE = 90.0f;   // degree



///////////////////////////////////////////////////////////////////////////////
class StlModel
{
public:
    StlModel();
    ~StlModel();

    // load stl file
    bool read(const char* file);
    bool save(const char* file, bool binary=true, const float* matrix=NULL);

    // copy vertex data to this object
    void copy(const float* vertices, unsigned int vertexCount,
              const float* normals, unsigned int normalCount,
              const unsigned int* indices, unsigned int indexCount);

    // re-generate and soften normals
    void smoothNormals(float smoothAngle = STL_SMOOTH_ANGLE);

    // remove duplicated vertices
    void removeDuplicates();

    // vertex attributes
    unsigned int getVertexCount() const         { return (unsigned int)vertices.size() / 3; }
    unsigned int getNormalCount() const         { return (unsigned int)normals.size() / 3; }
    unsigned int getIndexCount() const          { return (unsigned int)indices.size(); }  // total
    unsigned int getTriangleCount() const       { return (unsigned int)indices.size() / 3; }
    const BoundingBox& getBoundingBox() const   { return bound; }

    // return data as 1D array
    const float* getVertices() const            { return &vertices[0]; }
    const float* getNormals() const             { return &normals[0]; }
    const unsigned int* getIndices() const      { return &indices[0]; }

    // for interleaved vertices
    // NOTE: interleaved vertex array will be built automatically
    //       The stride, interleaved vertex count and size are only valid after
    //       buildInterleavedVertices() called
    const float* getInterleavedVertices();          // return interleaved data
    int getInterleavedStride() const                { return stride; }
    unsigned int getInterleavedVertexCount() const  { return getVertexCount(); }
    unsigned int getInterleavedVertexSize() const   { return (unsigned int)interleavedVertices.size() * sizeof(float); } // # of bytes

    // for path & file names
    const std::string& getStlFileName() const   { return stlFileName; }
    const std::string& getStlDirectory() const  { return stlDirectory; }

    const std::string& getErrorMessage() const  { return errorMessage; }
    void printSelf() const;

protected:


private:
    void init();
    bool readAscii(const char* file);
    bool readBinary(const char* file);
    bool saveAscii(const char* file, const float* matrix);
    bool saveBinary(const char* file, const float* matrix);
    void computeBoundingBox();
    Vector3 computeFaceNormal(const Vector3& v1, const Vector3& v2, const Vector3& v3);
    void splitFaces();
    void findDuplicates();
    void joinFaces();
    void averageNormals(float smoothAngle = STL_SMOOTH_ANGLE);
    void buildInterleavedVertices();             // create interleaved data
    float s2f(const std::string& str);

    // transform a vertex with 4x4 matrix: M*v
    Vector3 transform(const float* mat, const Vector3& vec);

    std::vector<float> vertices;                // vertex position array for opengl
    std::vector<float> normals;                 // vertex normal array for opengl
    std::vector<unsigned int> indices;          // index array for opengl
    std::vector<Vector3> faceNormals;           // normals per face
    std::vector<float> interleavedVertices;     // for opengl interleaved vertex

    // split vertex data without sharing vertices for smoothing normals
    std::vector<Vector3> splitVertices;
    std::vector<Vector3> splitNormals;
    std::multimap<Vector3, unsigned int> splitVertexMap;
    std::map<unsigned int, unsigned int> sharedVertexLookup;

    BoundingBox bound;

    int stride;                                 // # of bytes to hop to the next vertex

    std::string stlDirectory;                   // file location with trailing /
    std::string stlFileName;                    // file name without path

    std::string errorMessage;
};

#endif // STL_MODEL_H
