///////////////////////////////////////////////////////////////////////////////
// main.cpp
// ========
// testing StlModel class
//
//  AUTHOR: Song Ho Ahn (song.ahn@gmail.com)
// CREATED: 2025-09-20
// UPDATED: 2026-01-24
///////////////////////////////////////////////////////////////////////////////

// in order to get function prototypes from glext.h, define GL_GLEXT_PROTOTYPES before including glext.h
#define GL_GLEXT_PROTOTYPES

#ifdef __APPLE__
#include <GLUT/glut.h>
#else
#include <GL/glut.h>
#endif

#include <cstdlib>
#include <iostream>
#include <sstream>
#include <iomanip>
#include <vector>
#include <algorithm>
#include "glExtension.h"                        // OpenGL extension helper
#include "StlModel.h"
#include "Vectors.h"
#include "BoundingBox.h"
#include "Timer.h"
#include "geomUtils.h"

// GLUT CALLBACK functions
void displayCB();
void reshapeCB(int w, int h);
void timerCB(int millisec);
void idleCB();
void keyboardCB(unsigned char key, int x, int y);
void mouseCB(int button, int stat, int x, int y);
void mouseMotionCB(int x, int y);

// CALLBACK function when exit() called ///////////////////////////////////////
void exitCB();


void initGL();
int  initGLUT(int argc, char **argv);
bool initSharedMem();
void clearSharedMem();
void initLights();
void setCamera(float posX, float posY, float posZ, float targetX, float targetY, float targetZ);
GLuint createVBO(const void* data, int dataSize, GLenum target=GL_ARRAY_BUFFER, GLenum usage=GL_STATIC_DRAW);
void deleteVBO(const GLuint vboId);
void drawString(const char *str, int x, int y, float color[4], void *font);
void drawString3D(const char *str, float pos[3], float color[4], void *font);
void drawAxis(float sizeX=1.0f, float sizeY=1.0f, float sizeZ=1.0f);
void showInfo();
void showFPS();
void toOrtho();
void toPerspective();

// constants
const int   SCREEN_WIDTH    = 500;
const int   SCREEN_HEIGHT   = 500;
const float CAMERA_DISTANCE = 10.0f;
const int   TEXT_WIDTH      = 8;
const int   TEXT_HEIGHT     = 13;

// global variables
void *font = GLUT_BITMAP_8_BY_13;
int screenWidth;
int screenHeight;
GLuint vboId1, vboId2;                 // 1=interleaved, 2=non-interleaved
GLuint iboId;
bool mouseLeftDown;
bool mouseRightDown;
float mouseX, mouseY;
float cameraAngleX;
float cameraAngleY;
float cameraDistance;
bool vboSupported, vboUsed;
bool interleaveUsed;
int drawMode = 0;
bool axisVisible;
Timer timer;
float playTime;

// STL model
StlModel stl;
Vector3 center;
float boundingRadius;



///////////////////////////////////////////////////////////////////////////////
// draw the local axis of an object
///////////////////////////////////////////////////////////////////////////////
void drawAxis(float sizeX, float sizeY, float sizeZ)
{
    glDepthFunc(GL_ALWAYS);     // to avoid visual artifacts with grid lines
    glDisable(GL_LIGHTING);

    // draw axis
    glLineWidth(3);
    glBegin(GL_LINES);
        glColor3f(1, 0, 0);
        glVertex3f(0, 0, 0);
        glVertex3f(sizeX, 0, 0);
        glColor3f(0, 1, 0);
        glVertex3f(0, 0, 0);
        glVertex3f(0, sizeY, 0);
        glColor3f(0, 0, 1);
        glVertex3f(0, 0, 0);
        glVertex3f(0, 0, sizeZ);
    glEnd();
    glLineWidth(1);

    // draw arrows(actually big square dots)
    glPointSize(5);
    glBegin(GL_POINTS);
        glColor3f(1, 0, 0);
        glVertex3f(sizeX, 0, 0);
        glColor3f(0, 1, 0);
        glVertex3f(0, sizeY, 0);
        glColor3f(0, 0, 1);
        glVertex3f(0, 0, sizeZ);
    glEnd();
    glPointSize(1);

    // restore default settings
    glEnable(GL_LIGHTING);
    glDepthFunc(GL_LEQUAL);
}



///////////////////////////////////////////////////////////////////////////////
int main(int argc, char **argv)
{
    // init global vars
    initSharedMem();

    // init GLUT and GL
    initGLUT(argc, argv);
    initGL();

    // register exit callback
    atexit(exitCB);

    timer.start();
    if(argc == 2)
    {
        // load a STL model
        stl.read(argv[1]);
    }
    else
    {
        // load default model
        //stl.read("test_ascii.stl");
        //stl.read("test2_binary.stl");
        //stl.read("cube_ascii.stl");
        //stl.read("cube_binary.stl");
        //stl.read("duck_lowpoly_ascii.stl");
        //stl.read("duck_lowpoly_binary.stl");
        stl.read("debugger_50k.stl");
    }
    stl.printSelf();
    timer.stop();
    std::cout << "\nLoading Elapsed Time: " << timer.getElapsedTime() << " sec.\n" << std::endl;

    // remove duplicated vertics
    //stl.removeDuplicates();
    //stl.smoothNormals(60.0f);
    //std::cout << "After Removing Duplicates...\n";
    //stl.printSelf();

    /*
    //@@ test saving
    float mat[16];
    mat[0]=1; mat[1]=0; mat[2]=0; mat[3]=0;     // left
    mat[4]=0; mat[5]=0; mat[6]=-1; mat[7]=0;    // up
    mat[8]=0; mat[9]=1; mat[10]=0; mat[11]=0;   // forward
    mat[12]=0; mat[13]=0; mat[14]=0; mat[14]=1; // translate
    const float* vertices = stl.getVertices();
    const float* normals = stl.getNormals();
    const unsigned int* indices = stl.getIndices();
    unsigned int vertexCount = stl.getVertexCount();
    unsigned int normalCount = stl.getNormalCount();
    unsigned int indexCount = stl.getIndexCount();
    stl.copy(vertices, vertexCount, normals, normalCount, indices, indexCount);
    stl.save("test_ascii.stl", false);
    stl.save("test_binary.stl");
    stl.save("test2_binary.stl", true, mat);
    */

    std::cout << "Surface Area: " << computeArea(stl.getVertices(), stl.getIndices(), stl.getIndexCount()) << std::endl;
    std::cout << "Volume: " << computeVolume(stl.getVertices(), stl.getIndices(), stl.getIndexCount()) << std::endl;
   
    BoundingBox bb = stl.getBoundingBox();
    center.x = bb.getCenterX();
    center.y = bb.getCenterY();
    center.z = bb.getCenterZ();
    boundingRadius = bb.getRadius();
    cameraDistance = boundingRadius * 2.0f;
    std::cout << "Center: " << center << std::endl;
    std::cout << "Bounding Radius: " << boundingRadius << std::endl;
    std::cout << "Camera Dist: " << cameraDistance << std::endl;

    // get OpenGL extensions
    glExtension& ext = glExtension::getInstance();
    vboSupported = vboUsed = ext.isSupported("GL_ARB_vertex_buffer_object");
    if(vboSupported)
    {
        std::cout << "Video card supports GL_ARB_vertex_buffer_object." << std::endl;

        // interleaved method
        glGenBuffers(1, &vboId1);
        glBindBuffer(GL_ARRAY_BUFFER, vboId1);
        const float* interleavedVertices = stl.getInterleavedVertices();
        unsigned int dataSize = stl.getInterleavedVertexSize();
        glBufferData(GL_ARRAY_BUFFER, dataSize, interleavedVertices, GL_STATIC_DRAW);

        // non-interleaved method
        int vertexCount = stl.getVertexCount();
        int normalCount = stl.getNormalCount();
        const float* vertexData = stl.getVertices();
        const float* normalData = stl.getNormals();
        int vertexDataSize = vertexCount * 3 * sizeof(float);
        int normalDataSize = normalCount * 3 * sizeof(float);

        glGenBuffers(1, &vboId2);
        glBindBuffer(GL_ARRAY_BUFFER, vboId2);
        glBufferData(GL_ARRAY_BUFFER, vertexDataSize+normalDataSize, 0, GL_STATIC_DRAW);
        glBufferSubData(GL_ARRAY_BUFFER, 0, vertexDataSize, vertexData);                 // copy vertices starting from 0 offest
        glBufferSubData(GL_ARRAY_BUFFER, vertexDataSize, normalDataSize, normalData);    // copy normals after vertices

        // create a IBO and copy the indices to it
        glGenBuffers(1, &iboId);
        glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, iboId);
        glBufferData(GL_ELEMENT_ARRAY_BUFFER, stl.getIndexCount()*sizeof(int), (void*)stl.getIndices(), GL_STATIC_DRAW);
    }
    else
    {
        std::cout << "Video card does NOT support GL_ARB_vertex_buffer_object." << std::endl;
    }

    // disable V-Sync
#ifdef _WIN32
    if(ext.isSupported("WGL_EXT_swap_control"))
    {
        wglSwapIntervalEXT(0);
        std::cout << "Video card supports WGL_EXT_swap_control.\nDisable V-Sync." << std::endl;
    }
#endif

    // start timer
    timer.start();
    playTime = 0;

    // the last GLUT call (LOOP)
    // window will be shown and display callback is triggered by events
    // NOTE: this call never return main().
    glutMainLoop(); /* Start GLUT event-processing loop */

    return 0;
}



///////////////////////////////////////////////////////////////////////////////
// initialize GLUT for windowing
///////////////////////////////////////////////////////////////////////////////
int initGLUT(int argc, char **argv)
{
    // GLUT stuff for windowing
    // initialization openGL window.
    // it is called before any other GLUT routine
    glutInit(&argc, argv);

    glutInitDisplayMode(GLUT_RGBA | GLUT_DOUBLE | GLUT_DEPTH | GLUT_STENCIL);   // display mode

    glutInitWindowSize(screenWidth, screenHeight);  // window size

    glutInitWindowPosition(100, 100);               // window location

    // finally, create a window with openGL context
    // Window will not displayed until glutMainLoop() is called
    // it returns a unique ID
    int handle = glutCreateWindow(argv[0]);         // param is the title of window

    // register GLUT callback functions
    glutDisplayFunc(displayCB);
    //glutTimerFunc(33, timerCB, 33);                 // redraw only every given millisec
    glutIdleFunc(idleCB);                           // redraw when idle
    glutReshapeFunc(reshapeCB);
    glutKeyboardFunc(keyboardCB);
    glutMouseFunc(mouseCB);
    glutMotionFunc(mouseMotionCB);

    return handle;
}



///////////////////////////////////////////////////////////////////////////////
// initialize OpenGL
// disable unused features
///////////////////////////////////////////////////////////////////////////////
void initGL()
{
    glShadeModel(GL_SMOOTH);                    // shading mathod: GL_SMOOTH or GL_FLAT
    //glShadeModel(GL_FLAT);
    glPixelStorei(GL_UNPACK_ALIGNMENT, 4);      // 4-byte pixel alignment

    // enable /disable features
    glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST);
    //glHint(GL_LINE_SMOOTH_HINT, GL_NICEST);
    //glHint(GL_POLYGON_SMOOTH_HINT, GL_NICEST);
    glEnable(GL_DEPTH_TEST);
    glEnable(GL_LIGHTING);
    //glEnable(GL_TEXTURE_2D);
    glEnable(GL_CULL_FACE);

    // track material ambient and diffuse from surface color, call it before glEnable(GL_COLOR_MATERIAL)
    //glColorMaterial(GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE);
    //glEnable(GL_COLOR_MATERIAL);

    glClearColor(0, 0, 0, 0);                   // background color
    glClearStencil(0);                          // clear stencil buffer
    glClearDepth(1.0f);                         // 0 is near, 1 is far
    glDepthFunc(GL_LEQUAL);

    initLights();
}



///////////////////////////////////////////////////////////////////////////////
// write 2d text using GLUT
// The projection matrix must be set to orthogonal before call this function.
///////////////////////////////////////////////////////////////////////////////
void drawString(const char *str, int x, int y, float color[4], void *font)
{
    glPushAttrib(GL_LIGHTING_BIT | GL_CURRENT_BIT); // lighting and color mask
    glDisable(GL_LIGHTING);     // need to disable lighting for proper text color
    glDisable(GL_TEXTURE_2D);

    glColor4fv(color);          // set text color
    glRasterPos2i(x, y);        // place text position

    // loop all characters in the string
    while(*str)
    {
        glutBitmapCharacter(font, *str);
        ++str;
    }

    glEnable(GL_TEXTURE_2D);
    glEnable(GL_LIGHTING);
    glPopAttrib();
}



///////////////////////////////////////////////////////////////////////////////
// draw a string in 3D space
///////////////////////////////////////////////////////////////////////////////
void drawString3D(const char *str, float pos[3], float color[4], void *font)
{
    glPushAttrib(GL_LIGHTING_BIT | GL_CURRENT_BIT); // lighting and color mask
    glDisable(GL_LIGHTING);     // need to disable lighting for proper text color
    glDisable(GL_TEXTURE_2D);

    glColor4fv(color);          // set text color
    glRasterPos3fv(pos);        // place text position

    // loop all characters in the string
    while(*str)
    {
        glutBitmapCharacter(font, *str);
        ++str;
    }

    glEnable(GL_TEXTURE_2D);
    glEnable(GL_LIGHTING);
    glPopAttrib();
}



///////////////////////////////////////////////////////////////////////////////
// initialize global variables
///////////////////////////////////////////////////////////////////////////////
bool initSharedMem()
{
    screenWidth = SCREEN_WIDTH;
    screenHeight = SCREEN_HEIGHT;

    mouseLeftDown = mouseRightDown = false;
    mouseX = mouseY = 0;

    cameraAngleX = cameraAngleY = 0;
    cameraDistance = CAMERA_DISTANCE;
    boundingRadius = 0;

    drawMode = 0; // 0:fill, 1: wireframe, 2:points
    axisVisible = true;

    vboId1 = vboId2 = iboId = 0;
    vboSupported = vboUsed = false;
    interleaveUsed = true;

    return true;
}



///////////////////////////////////////////////////////////////////////////////
// clean up shared memory
///////////////////////////////////////////////////////////////////////////////
void clearSharedMem()
{
    // clean up VBOs
    if(vboSupported)
    {
        deleteVBO(vboId1);
        deleteVBO(vboId2);
        deleteVBO(iboId);
        vboId1 = vboId2 = iboId = 0;
    }
}



///////////////////////////////////////////////////////////////////////////////
// initialize lights
///////////////////////////////////////////////////////////////////////////////
void initLights()
{
    // set up light colors (ambient, diffuse, specular)
    GLfloat lightKa[] = {.0f, .0f, .0f, 1.0f};  // ambient light
    GLfloat lightKd[] = {.7f, .7f, .7f, 1.0f};  // diffuse light
    GLfloat lightKs[] = {1, 1, 1, 1};           // specular light
    glLightfv(GL_LIGHT0, GL_AMBIENT, lightKa);
    glLightfv(GL_LIGHT0, GL_DIFFUSE, lightKd);
    glLightfv(GL_LIGHT0, GL_SPECULAR, lightKs);

    // position the light
    //float lightPos[4] = {0, 0, 20, 1}; // positional light
    float lightPos[4] = {0, 0, 1, 0}; // directional light
    glLightfv(GL_LIGHT0, GL_POSITION, lightPos);

    glEnable(GL_LIGHT0);                        // MUST enable each light source after configuration
}



///////////////////////////////////////////////////////////////////////////////
// set camera position and lookat direction
///////////////////////////////////////////////////////////////////////////////
void setCamera(float posX, float posY, float posZ, float targetX, float targetY, float targetZ)
{
    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();
    gluLookAt(posX, posY, posZ, targetX, targetY, targetZ, 0, 1, 0); // eye(x,y,z), focal(x,y,z), up(x,y,z)
}



///////////////////////////////////////////////////////////////////////////////
// generate vertex buffer object and bind it with its data
// You must give 2 hints about data usage; target and mode, so that OpenGL can
// decide which data should be stored and its location.
// VBO works with 2 different targets; GL_ARRAY_BUFFER for vertex arrays
// and GL_ELEMENT_ARRAY_BUFFER for index array in glDrawElements().
// The default target is GL_ARRAY_BUFFER.
// By default, usage mode is set as GL_STATIC_DRAW.
// Other usages are GL_STREAM_DRAW, GL_STREAM_READ, GL_STREAM_COPY,
// GL_STATIC_DRAW, GL_STATIC_READ, GL_STATIC_COPY,
// GL_DYNAMIC_DRAW, GL_DYNAMIC_READ, GL_DYNAMIC_COPY.
///////////////////////////////////////////////////////////////////////////////
GLuint createVBO(const void* data, int dataSize, GLenum target, GLenum usage)
{
    GLuint id = 0;  // 0 is reserved, glGenBuffers() will return non-zero id if success

    glGenBuffers(1, &id);                           // create a vbo
    glBindBuffer(target, id);                       // activate vbo id to use
    glBufferData(target, dataSize, data, usage);    // upload data to video card

    // check data size in VBO is same as input array, if not return 0 and delete VBO
    int bufferSize = 0;
    glGetBufferParameteriv(target, GL_BUFFER_SIZE, &bufferSize);
    if(dataSize != bufferSize)
    {
        glDeleteBuffers(1, &id);
        id = 0;
        std::cout << "[createVBO()] Data size is mismatch with input array\n";
    }

    return id;      // return VBO id
}



///////////////////////////////////////////////////////////////////////////////
// destroy a VBO
// If VBO id is not valid or zero, then OpenGL ignores it silently.
///////////////////////////////////////////////////////////////////////////////
void deleteVBO(const GLuint vboId)
{
    glDeleteBuffers(1, &vboId);
}



///////////////////////////////////////////////////////////////////////////////
// display info messages
///////////////////////////////////////////////////////////////////////////////
void showInfo()
{
    // backup current model-view matrix
    glPushMatrix();                     // save current modelview matrix
    glLoadIdentity();                   // reset modelview matrix

    // set to 2D orthogonal projection
    glMatrixMode(GL_PROJECTION);        // switch to projection matrix
    glPushMatrix();                     // save current projection matrix
    glLoadIdentity();                   // reset projection matrix
    gluOrtho2D(0, screenWidth, 0, screenHeight); // set to orthogonal projection

    float color[4] = {1, 1, 1, 1};

    std::stringstream ss;
    ss << "VBO: " << (vboUsed ? "on" : "off") << std::ends;  // add 0(ends) at the end
    drawString(ss.str().c_str(), 1, screenHeight-TEXT_HEIGHT, color, font);
    ss.str(""); // clear buffer

    ss << "Interleaved: " << (interleaveUsed ? "on" : "off") << std::ends;
    drawString(ss.str().c_str(), 1, screenHeight-(TEXT_HEIGHT*2), color, font);
    ss.str(""); // clear buffer

    ss << "Press SPACE key to toggle VBO, I to toggle interleave mode." << std::ends;
    drawString(ss.str().c_str(), 1, 1, color, font);

    // restore projection matrix
    glPopMatrix();                   // restore to previous projection matrix

    // restore modelview matrix
    glMatrixMode(GL_MODELVIEW);      // switch to modelview matrix
    glPopMatrix();                   // restore to previous modelview matrix
}



///////////////////////////////////////////////////////////////////////////////
// display frame rates
///////////////////////////////////////////////////////////////////////////////
void showFPS()
{
    static Timer timer;
    static int count = 0;
    static std::string fps = "0.0 FPS";
    double elapsedTime = 0.0;;

    ++count;

    // backup current model-view matrix
    glPushMatrix();                     // save current modelview matrix
    glLoadIdentity();                   // reset modelview matrix

    // set to 2D orthogonal projection
    glMatrixMode(GL_PROJECTION);        // switch to projection matrix
    glPushMatrix();                     // save current projection matrix
    glLoadIdentity();                   // reset projection matrix
    gluOrtho2D(0, screenWidth, 0, screenHeight); // set to orthogonal projection

    float color[4] = {1, 1, 0, 1};

    // update fps every second
    elapsedTime = timer.getElapsedTime();
    if(elapsedTime >= 1.0)
    {
        std::stringstream ss;
        ss << std::fixed << std::setprecision(1);
        ss << (count / elapsedTime) << " FPS" << std::ends; // update fps string
        ss << std::resetiosflags(std::ios_base::fixed | std::ios_base::floatfield);
        fps = ss.str();

        //DEBUG
        //std::cout << "Avg frame time: " << elapsedTime / count * 1000.0 << " ms" << std::endl;

        count = 0;                      // reset counter
        timer.start();                  // restart timer
    }
    int textWidth = (int)fps.size() * TEXT_WIDTH;
    drawString(fps.c_str(), screenWidth-textWidth, screenHeight-TEXT_HEIGHT, color, font);

    // restore projection matrix
    glPopMatrix();                      // restore to previous projection matrix

    // restore modelview matrix
    glMatrixMode(GL_MODELVIEW);         // switch to modelview matrix
    glPopMatrix();                      // restore to previous modelview matrix

}



///////////////////////////////////////////////////////////////////////////////
// set projection matrix as orthogonal
///////////////////////////////////////////////////////////////////////////////
void toOrtho()
{
    // set viewport to be the entire window
    glViewport(0, 0, (GLsizei)screenWidth, (GLsizei)screenHeight);

    // set orthographic viewing frustum
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    glOrtho(0, screenWidth, 0, screenHeight, -1, 1);

    // switch to modelview matrix in order to set scene
    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();
}



///////////////////////////////////////////////////////////////////////////////
// set the projection matrix as perspective
///////////////////////////////////////////////////////////////////////////////
void toPerspective()
{
    // set viewport to be the entire window
    glViewport(0, 0, (GLsizei)screenWidth, (GLsizei)screenHeight);

    // set perspective viewing frustum
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    //gluPerspective(60.0f, (float)(screenWidth)/screenHeight, 0.1f, 1000.0f); // FOV, AspectRatio, NearClip, FarClip
    gluPerspective(60.0f, (float)(screenWidth)/screenHeight, boundingRadius * 0.1f, boundingRadius * 1000.0f); // FOV, AspectRatio, NearClip, FarClip

    // switch to modelview matrix in order to set scene
    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();
}





//=============================================================================
// CALLBACKS
//=============================================================================

void displayCB()
{
    static float angle = 0;
    float elapsedTime = (float)timer.getElapsedTimeInMilliSec();
    float frameTime = elapsedTime - playTime;
    playTime = elapsedTime;
    angle += frameTime * 0.02f;

    // clear buffer
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT | GL_STENCIL_BUFFER_BIT);

    // save the initial ModelView matrix before modifying ModelView matrix
    glPushMatrix();

    // tramsform camera
    glTranslatef(0, 0, -cameraDistance);
    glRotatef(cameraAngleX, 1, 0, 0);   // pitch
    glRotatef(cameraAngleY + angle, 0, 1, 0);   // heading

    // STL model is z-axis up
    glRotatef(90, 1,0,0);
    glRotatef(180, 0,1,0);

    // transform mesh
    glTranslatef(-center.x, -center.y, -center.z);

    // draw local axis
    if(axisVisible)
    {
        const float SCALE = 1.2f;
        BoundingBox bb = stl.getBoundingBox();
        drawAxis(bb.getRadiusX()*SCALE, bb.getRadiusY()*SCALE, bb.getRadiusZ()*SCALE*2);
    }

    if(vboUsed) // draw  object using VBO
    {
        // enable vertex arrays
        glEnableClientState(GL_NORMAL_ARRAY);
        glEnableClientState(GL_VERTEX_ARRAY);

        // interleaved mode
        if(interleaveUsed)
        {
            // bind VBOs with IDs and set the buffer offsets of the bound VBOs
            // When buffer object is bound with its ID, all pointers in gl*Pointer()
            // are treated as offset instead of real pointer.
            glBindBuffer(GL_ARRAY_BUFFER, vboId1);

            // before draw, specify vertex and index arrays with their offsets and stride
            int stride = stl.getInterleavedStride();
            glNormalPointer(GL_FLOAT, stride, (void*)(sizeof(float)*3));
            glVertexPointer(3, GL_FLOAT, stride, 0);
        }
        else
        {
            // non-interleaved method
            glBindBuffer(GL_ARRAY_BUFFER, vboId2);

            // before draw, specify vertex and index arrays with their offsets
            int normalOffset = stl.getVertexCount() * 3 * sizeof(float);
            glNormalPointer(GL_FLOAT, 0, (void*)normalOffset);
            glVertexPointer(3, GL_FLOAT, 0, 0);
        }

        glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, iboId);
        glDrawElements(GL_TRIANGLES, stl.getIndexCount(), GL_UNSIGNED_INT, 0);

        glDisableClientState(GL_VERTEX_ARRAY);  // disable vertex arrays
        glDisableClientState(GL_NORMAL_ARRAY);

        // it is good idea to release VBOs with ID 0 after use.
        // Once bound with 0, all pointers in gl*Pointer() behave as real
        // pointer, so, normal vertex array operations are re-activated
        glBindBuffer(GL_ARRAY_BUFFER, 0);
        glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, 0);
    }

    else        // draw object using vertex array method
    // notice that only difference between VBO and VA is binding buffers and offsets
    {
        // enable vertex arrays
        glEnableClientState(GL_NORMAL_ARRAY);
        glEnableClientState(GL_VERTEX_ARRAY);

        if(interleaveUsed)
        {
            // before draw, specify vertex arrays
            const float* vertices = stl.getInterleavedVertices();
            int stride = stl.getInterleavedStride();
            glVertexPointer(3, GL_FLOAT, stride, vertices);
            glNormalPointer(GL_FLOAT, stride, vertices + 3);
        }
        else
        {
            // non-interleaved method
            // before draw, specify vertex arrays
            glNormalPointer(GL_FLOAT, 0, stl.getNormals());
            glVertexPointer(3, GL_FLOAT, 0, stl.getVertices());
        }

        glDrawElements(GL_TRIANGLES, stl.getIndexCount(), GL_UNSIGNED_INT, stl.getIndices());

        glDisableClientState(GL_VERTEX_ARRAY);  // disable vertex arrays
        glDisableClientState(GL_NORMAL_ARRAY);
    }

    // draw info messages
    showInfo();
    showFPS();

    glPopMatrix();

    glutSwapBuffers();
}


void reshapeCB(int w, int h)
{
    screenWidth = w;
    screenHeight = h;
    toPerspective();
}


void timerCB(int millisec)
{
    glutTimerFunc(millisec, timerCB, millisec);
    glutPostRedisplay();
}


void idleCB()
{
    glutPostRedisplay();
}


void keyboardCB(unsigned char key, int x, int y)
{
    switch(key)
    {
    case 27: // ESCAPE
        exit(0);
        break;

    case ' ':
        if(vboSupported)
            vboUsed = !vboUsed;
        break;

    case 'a':
    case 'A':
        axisVisible = !axisVisible;
        break;

    case 'd': // switch rendering modes (fill -> wire -> point)
    case 'D':
        drawMode++;
        drawMode = drawMode % 3;
        if(drawMode == 0)        // fill mode
        {
            glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
            glEnable(GL_DEPTH_TEST);
            glEnable(GL_CULL_FACE);
        }
        else if(drawMode == 1)  // wireframe mode
        {
            glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
            glDisable(GL_DEPTH_TEST);
            glDisable(GL_CULL_FACE);
        }
        else                    // point mode
        {
            glPolygonMode(GL_FRONT_AND_BACK, GL_POINT);
            glDisable(GL_DEPTH_TEST);
            glDisable(GL_CULL_FACE);
        }
        break;

    case 'i':
    case 'I':
        interleaveUsed = !interleaveUsed;
        break;

    default:
        ;
    }
}


void mouseCB(int button, int state, int x, int y)
{
    mouseX = x;
    mouseY = y;

    if(button == GLUT_LEFT_BUTTON)
    {
        if(state == GLUT_DOWN)
        {
            mouseLeftDown = true;
        }
        else if(state == GLUT_UP)
            mouseLeftDown = false;
    }

    else if(button == GLUT_RIGHT_BUTTON)
    {
        if(state == GLUT_DOWN)
        {
            mouseRightDown = true;
        }
        else if(state == GLUT_UP)
            mouseRightDown = false;
    }
}


void mouseMotionCB(int x, int y)
{
    if(mouseLeftDown)
    {
        cameraAngleY += (x - mouseX);
        cameraAngleX += (y - mouseY);
        mouseX = x;
        mouseY = y;
    }
    if(mouseRightDown)
    {
        cameraDistance -= (y - mouseY) * (boundingRadius * 0.1f);
        if(cameraDistance > boundingRadius * 10)
            cameraDistance = boundingRadius * 10.0f;
        else if(cameraDistance < boundingRadius)
            cameraDistance = boundingRadius;

        mouseY = y;
    }
}



void exitCB()
{
    clearSharedMem();
}
